//
//  XWIFIManager.h
//  Printer
//

#import <Foundation/Foundation.h>
#import <Cocoa/Cocoa.h>
#import "NetProfile.h"

@class XWIFIManager;
@protocol XWIFIManagerDelegate <NSObject>
@optional

/// Callback for successful connection
///
/// @param host Host address
/// @param port Port number
- (void)xWifiConnectedToHost:(NSString *)host port:(UInt16)port;

/// Callback for disconnection error
/// @param error Error information
- (void)xWifiDisconnectWithError:(NSError *)error host:(NSString *)host;

/// Callback for successful data transmission
/// @param tag Tag value
- (void)xWifiWriteValueWithTag:(long)tag host:(NSString *)host;

/// Callback for receiving data from the printer
/// @param data Received data
- (void)xWifiReceiveValueForData:(NSData *)data host:(NSString *)host;
@end

/// Definition of the callback block type for receiving data.
typedef void (^XWIFIManagerReceiveBlock)(NSData *data);

/// Definition of the callback block type for writing data.
typedef void (^XWIFIManagerWriteBlock)(BOOL success, NSError *error);

/// Definition of the callback block type for reporting POS printer status.
typedef void (^XWIFIPOSPrinterStatusBlock)(NSData *status);

/// Definition of the callback block type for reporting label printer status.
typedef void (^XWIFILabelPrinterStatusBlock)(NSData *status);

/// Definition of the callback block type for reporting the printer's serial number.
typedef void (^XWIFIPrinterSNBlock)(NSString *sn);

/// Definition of the callback block type for reporting the printer's check status.
typedef void (^XWIFIPrinterCheckBlock)(NSData *check);

/// Definition of the callback block type for reporting a found printer's information.
typedef void (^XWIFIManagerFoundPrinterBlock)(NetProfile *foundPrinter);

/// Definition of the callback block type for reporting cash drawer status.
typedef void (^XWIFICashBoxBlock)(NSData *status);

@class NetProfile;
@interface XWIFIManager : NSObject
// host
@property (nonatomic, copy) NSString *hostStr;
// port
@property (nonatomic, assign) UInt16 port;
// whether connect
@property (nonatomic, assign) BOOL isConnect;

/// The delegate that receives WIFI manager events.
@property (nonatomic, weak) id<XWIFIManagerDelegate> delegate;

/// Callback block called when receiving data from the Wi-Fi manager.
@property (nonatomic, copy) XWIFIManagerReceiveBlock receiveBlock;

/// Callback block called when writing data through the Wi-Fi manager.
@property (nonatomic, copy) XWIFIManagerWriteBlock writeBlock;

/// Callback block called when a printer is found by the Wi-Fi manager.
@property (nonatomic, copy) XWIFIManagerFoundPrinterBlock foundPrinterBlock;

/// Callback block called when reporting POS printer status.
@property (nonatomic, copy) XWIFIPOSPrinterStatusBlock statusPOSBlock;

/// Callback block called when reporting label printer status.
@property (nonatomic, copy) XWIFILabelPrinterStatusBlock statusLabelBlock;

/// Callback block called when a printer's serial number (SN) is available.
@property (nonatomic, copy) XWIFIPrinterSNBlock snBlock;

/// Callback block called when a printer check operation completes.
@property (nonatomic, copy) XWIFIPrinterCheckBlock checkBlock;

/// Callback block called when reporting cash drawer status.
@property (nonatomic, copy) XWIFICashBoxBlock cashBoxBlock;

@property (nonatomic, strong) NetProfile *connectedPrinter;

/// Get the singleton instance
+ (instancetype)sharedInstance;

/// Remove a delegate object
/// @param delegate Delegate object
- (void)removeDelegate:(id<XWIFIManagerDelegate>) delegate;

/// Remove all delegate objects
- (void)removeAllDelegates;

/// Connect to the printer address
/// @param hostStr Printer host address
/// @param port Port number
- (void)connectWithHost:(NSString *)hostStr port:(UInt16)port;

/// Disconnect
- (void)disconnect;

/// Write commands to the printer
/// @param data Command data
- (void)writeCommandWithData:(NSData *)data;

/// Write commands to the printer and specify a receive callback
/// @param data Command data
/// @param receiveBlock Receive callback
- (void)writeCommandWithData:(NSData *)data receiveCallBack:(XWIFIManagerReceiveBlock)receiveBlock;

/// Write commands to the printer and specify a write completion callback
/// @param data Command data
/// @param writeBlock Write completion callback
- (void)writeCommandWithData:(NSData *)data writeCallBack:(XWIFIManagerWriteBlock)writeBlock;

/// Create a UDP Socket
/// @return Whether the creation was successful
- (BOOL)createUdpSocket;

/// Close the UDP Socket
- (void)closeUdpSocket;

/// Search for printers
/// @param foundPrinterBlock Callback when a printer is found
- (void)sendFindCmd:(XWIFIManagerFoundPrinterBlock)foundPrinterBlock;

/// Set IP configuration
/// @param ip IP address
/// @param mask Subnet mask
/// @param gateway Gateway
/// @param dhcp Whether to enable DHCP
- (void)setIPConfigWithIP:(NSString *)ip Mask:(NSString *)mask Gateway:(NSString *)gateway DHCP:(BOOL)dhcp;

/// Set WiFi configuration
///
/// @param ip IP address
/// @param mask Subnet mask
/// @param gateway Gateway
/// @param ssid WiFi SSID
/// @param password WiFi password
/// @param encrypt Encryption method
- (void)setWiFiConfigWithIP:(NSString *)ip mask:(NSString *)mask gateway:(NSString *)gateway ssid:(NSString *)ssid password:(NSString *)password encrypt:(NSUInteger)encrypt;

/// Check printer connection status
/// @return Whether connected
- (BOOL)printerIsConnect;

/// Get printer serial number
/// @param snBlock Serial number callback
- (void)printerSN:(XWIFIPrinterSNBlock)snBlock;

/// Get copyright information
/// @return Copyright information string
+ (NSString *)GetCopyRight;

/// Printer status (for receipt printers)
/// @param statusBlock Status callback
- (void)printerPOSStatus:(XWIFIPOSPrinterStatusBlock)statusBlock;

/// Printer status (for label printers)
/// @param statusBlock Status callback
- (void)printerLabelStatus:(XWIFILabelPrinterStatusBlock)statusBlock;

/// Query printer status
/// @param type Query type (1: common status, 2: offline status, 3: error status, 4: paper feed status)
/// @param checkBlock Query result callback
- (void)printerPOSCheck:(int)type checkBlock:(XWIFIPrinterCheckBlock)checkBlock;

/// Cash drawer status
/// @param cashBoxBlock Status callback
- (void)cashBoxCheck:(XWIFICashBoxBlock)cashBoxBlock;

@end
